#include "SoundManager.h"
#include <assert.h>

#include <stdio.h>
#include <algorithm>


SoundManager::SoundManager()
 : m_system(NULL),
   m_channel(0),
   m_sound(NULL),
   m_numtags(0),
   m_numtagsupdated(0),
   m_count(0),
   m_version(0),
   m_fOriginalFrequency(-1.f)
{
    std::fill( m_afSpectrum, m_afSpectrum + ms_nSpectrumSize, 0.f );
}


SoundManager::~SoundManager()
{
    if ( m_sound )
        m_sound->release();
    if ( m_system )
    {
        m_system->close();
        m_system->release();
    }
}

bool SoundManager::Init()
{
    FMOD_RESULT result;

    FMOD::System* pSystem;
    result = FMOD::System_Create(&pSystem);
    if (result != FMOD_OK )
    {
        printf("FMOD error! (%d) %s\n", result, FMOD_ErrorString(result));
        return false;
    }

    unsigned int version = 0;
    result = pSystem->getVersion(&version);
    if (result != FMOD_OK )
    {
        printf("FMOD error! (%d) %s\n", result, FMOD_ErrorString(result));
        return false;
    }

    if (version < FMOD_VERSION)
    {
        printf("Error!  You are using an old version of FMOD %08x.  This program requires %08x\n", version, FMOD_VERSION);
        return false;
    }

    result = pSystem->init(8, FMOD_INIT_NORMAL, 0);
    if (result != FMOD_OK )
    {
        printf("FMOD error! (%d) %s\n", result, FMOD_ErrorString(result));
        return false;
    }
    m_system = pSystem;
    m_version = version;
    return true;
}


bool SoundManager::PlaySound( const char* filePath )
{
    FMOD_RESULT result;

    // Open the specified file. Use FMOD_CREATESTREAM and FMOD_OPENONLY so it opens quickly
    FMOD::Sound* sound;
    result = m_system->createSound( filePath, FMOD_CREATESTREAM | FMOD_OPENONLY | FMOD_SOFTWARE | FMOD_LOOP_OFF | FMOD_2D,0, &sound);
    if (result != FMOD_OK )
    {
        printf("FMOD error! (%d) %s\n", result, FMOD_ErrorString(result));
        return false;
    }
    m_sound = sound;

    result = m_system->playSound(FMOD_CHANNEL_FREE, sound, false, &m_channel);

    // Get the original frequency so we know what to do with scale factor
    result = m_channel->getFrequency(&m_fOriginalFrequency);
    return true;
}



void SoundManager::Update()
{
    m_system->update();
    
    // Simple spectrum analysis
    m_channel->getSpectrum( m_afSpectrum, ms_nSpectrumSize, 0, FMOD_DSP_FFT_WINDOW_TRIANGLE );

    // Last value seems to be forgotten
    m_afSpectrum[ms_nSpectrumSize-1] = 0;

    // Normalize (make max = 1)
    float maxVal = m_afSpectrum[0];
    for ( int i = 1; i < ms_nSpectrumSize; ++i )
        maxVal = std::max(maxVal, m_afSpectrum[i]);

    float divMaxVal = float(1.0 / maxVal);
    for ( int i = 0; i < ms_nSpectrumSize; ++i )
        m_afSpectrum[i] *= divMaxVal;
}


void SoundManager::UpdateBuffer(float* destination, size_t destinationSize)
{
    size_t sourceSize = ms_nSpectrumSize * sizeof(float);
    assert(destinationSize >= sourceSize);
    memcpy(destination, &m_afSpectrum, sourceSize);
    
  
}

double SoundManager::GetTimeSec() const
{
    FMOD_RESULT result;
    unsigned int ms = 0;
    result = m_channel->getPosition(&ms, FMOD_TIMEUNIT_MS);
    if ((result != FMOD_OK) && (result != FMOD_ERR_INVALID_HANDLE) && (result != FMOD_ERR_CHANNEL_STOLEN))
        return 0;
    return double(ms) / 1000.0;
}

bool SoundManager::SetTimeSec( double tim )
{
    unsigned int ms = unsigned int (tim * 1000.0);
    FMOD_RESULT result = m_channel->setPosition(ms, FMOD_TIMEUNIT_MS);
    if ((result != FMOD_OK) && (result != FMOD_ERR_INVALID_HANDLE) && (result != FMOD_ERR_CHANNEL_STOLEN))
        return false;
    return true;
}

float SoundManager::GetScaleFactor() const
{
    float freq = 0.f;
    FMOD_RESULT result = m_channel->getFrequency(&freq);
    if ((result != FMOD_OK) && (result != FMOD_ERR_INVALID_HANDLE) && (result != FMOD_ERR_CHANNEL_STOLEN))
        return 0.f;
    return freq / m_fOriginalFrequency;
}

bool SoundManager::SetScaleFactor( float factor )
{
    
    FMOD_RESULT result = m_channel->setFrequency(factor * m_fOriginalFrequency);
    if ((result != FMOD_OK) && (result != FMOD_ERR_INVALID_HANDLE) && (result != FMOD_ERR_CHANNEL_STOLEN))
        return false;
    return true;
}

